#!/bin/bash

# set -e - 遇到错误立即退出
# set -u - 使用未定义的变量时报错
# set -x - 显示执行的命令（调试用）
# set -o pipefail - 管道中任何命令失败都返回失败
set -e

# config 目录,是基于当前 Jenkinsfile 的相对目录
# 检查并复制配置文件
haveConfig=false
if [ -d "./${CONFIG_DIR}" ] && [ -n "$(ls -A ./${CONFIG_DIR}/*.* 2>/dev/null)" ]; then
    cp -f ./${CONFIG_DIR}/*.* ./${BUILD_TARGET_DIR}
    echo "配置文件已复制"
    haveConfig=true
else
    echo "WARNING: ${CONFIG_DIR} 目录不存在或为空，跳过配置文件复制"
fi

# 进入构建目录
cd ${BUILD_TARGET_DIR}

# 1. 在 BUILD_TARGET_DIR 目录下创建 CONFIG_DIR 目录
mkdir -p ${CONFIG_DIR}

# 2. 拷贝配置文件
echo "拷贝 yml/yaml 配置文件"
if [ "${haveConfig}" = "true" ]; then
    # 如果有设置配置文件, 则直接使用
    # 注意：此时已经在 BUILD_TARGET_DIR 目录下，所以直接用 ./*.yml 和 ./*.yaml
    echo "使用设置的配置文件"
    copied=false
    if ls ./*.yml 1> /dev/null 2>&1; then
        cp -f ./*.yml ${CONFIG_DIR}/
        copied=true
    fi
    if ls ./*.yaml 1> /dev/null 2>&1; then
        cp -f ./*.yaml ${CONFIG_DIR}/
        copied=true
    fi
    if [ "${copied}" = "true" ]; then
        echo "已拷贝 yml/yaml 配置文件到 ${CONFIG_DIR} 目录"
    else
        echo "WARNING: 在 ${BUILD_TARGET_DIR} 目录下未找到 yml/yaml 文件"
    fi
else
    # 如果没有设置配置文件,则从 classes 目录下拷贝
    echo "没有设置配置文件,从 classes 目录下拷贝"
    if [ -d "classes" ] && [ -n "$(find classes -name '*.yml' -o -name '*.yaml')" ]; then
        find classes -name '*.yml' -o -name '*.yaml' | xargs -I {} cp {} ${CONFIG_DIR}/
        echo "配置文件已拷贝到 ${CONFIG_DIR} 目录"
    else
        echo "WARNING: 未找到 classes 目录或 yml 文件"
    fi
fi

# 打包的命令字符串
includeTargets="*.jar ${CONFIG_DIR}"

# 3. 将 static 文件夹拷贝到 BUILD_TARGET_DIR 目录下
echo "拷贝 static 文件, 从 ${STATIC_DIR} 下的所有文件到 ${BUILD_TARGET_DIR}"
if [ -n "${STATIC_DIR}" ] && [ -d "../${STATIC_DIR}" ]; then
    cp -r ../${STATIC_DIR} .
    echo "静态资源已拷贝到 ${BUILD_TARGET_DIR}/static 目录"
    includeTargets="${includeTargets} static"
else
    echo "WARNING: 未找到 static 目录"
fi

# 4. 还需要将其他 Jenkins 任务目录下的文件拷贝到 BUILD_TARGET_DIR
echo "拷贝 awdb 文件, 从 ${DB_AWDB_DIR} 下的所有文件到 ${BUILD_TARGET_DIR}"
if [ -n "${DB_AWDB_DIR}" ] && [ -d "${DB_AWDB_DIR}" ]; then
    echo "开始拷贝（强制覆盖）"
    cp -af "${DB_AWDB_DIR}/" . 2>/dev/null || true
    includeTargets="${includeTargets} *.awdb *.dat"
else
    echo "WARNING: DB_AWDB_DIR 未配置或目录不存在，跳过拷贝"
fi

echo "拷贝 ipdb 文件,从 ${DB_SOURCE_DIR} 下的所有文件到 ${BUILD_TARGET_DIR}"
if [ -n "${DB_SOURCE_DIR}" ] && [ -d "${DB_SOURCE_DIR}" ]; then
    echo "开始拷贝（强制覆盖）"
    cp -af "${DB_SOURCE_DIR}/" . 2>/dev/null || true
    includeTargets="${includeTargets} *.ipdb"
else
    echo "WARNING: DB_SOURCE_DIR 未配置或目录不存在，跳过拷贝"
fi

# 5. 打包
echo "打包文件: ${includeTargets}"
tar -zcf ${TAR_NAME} --exclude=.git/ --exclude=.gitignore ${includeTargets}

# 验证打包文件
if [ -f "${TAR_NAME}" ]; then
    echo "打包成功: ${TAR_NAME}"
    ls -lh ${TAR_NAME}
else
    echo "ERROR: 打包失败"
    exit 1
fi

